<?php

define("DEFINE_MY_ACCESS", true);
define("DEFINE_DHRU_FILE", true);

require_once __DIR__ . '/../../../comm.php';
require_once __DIR__ . '/../../../includes/fun.inc.php';
require_once __DIR__ . '/../../../includes/gateway.fun.php';
require_once __DIR__ . '/../../../includes/invoice.fun.php';

class UddoktaPay
{
    public $isActive;
    public $systemUrl;
    private static $instance;
    private $gatewayModuleName;
    private $gatewayParams;
    private $convoRate;
    private $client;
    private $invoice;
    private $due;
    private $total;
    private $baseUrl;
    private $credential;

    private function __construct()
    {
        $this->setRequest();
        $this->setGateway();
        $this->setInvoice();
    }

    public static function init()
    {
        if (self::$instance == null) {
            self::$instance = new UddoktaPay;
        }

        return self::$instance;
    }

    private function setGateway()
    {
        $this->gatewayModuleName = basename(__FILE__, '.php');
        $this->gatewayParams = loadGatewayModule($this->gatewayModuleName);
        $this->systemUrl = $this->gatewayParams['systemurl'];
        $this->isActive = !empty($this->gatewayParams['active']);
        $this->baseUrl = $this->normalizeBaseURL($this->gatewayParams['api_url']);

        $this->credential = [
            'api_key' => $this->gatewayParams['api_key'],
            'exchange_rate' => $this->gatewayParams['exchange_rate'],
        ];
    }

    private function normalizeBaseURL($apiBaseURL)
    {
        $baseURL = rtrim($apiBaseURL, '/');
        $apiSegmentPosition = strpos($baseURL, '/api');

        if ($apiSegmentPosition !== false) {
            $baseURL = substr($baseURL, 0, $apiSegmentPosition + 4); // Include '/api'
        }

        return $baseURL;
    }

    private function buildURL($endpoint)
    {
        $endpoint = ltrim($endpoint, '/');
        return $this->baseUrl . '/' . $endpoint;
    }

    private function setRequest()
    {
        $this->request = $_REQUEST;
    }

    private function setInvoice()
    {
        $result = select_query("tbl_invoices", "", ["id" => $this->request['id']]);
        $this->invoice = mysqli_fetch_assoc($result);

        $this->setCurrency();
        $this->setClient();
        $this->setDue();
        $this->setFee();
        $this->setTotal();
    }

    private function setCurrency()
    {
        $curcode = $this->gatewayParams['curcode'];
        if (!empty($curcode) && ($curcode !== 'BDT')) {
            $this->convoRate = $this->gatewayParams['exchange_rate'];
        } else {
            $this->convoRate = 1;
        }
    }

    private function setClient()
    {
        $result = select_query("tblUsers", "", ["id" => $this->invoice['userid']]);
        $this->client = mysqli_fetch_assoc($result);
    }

    private function setDue()
    {
        $this->due = $this->invoice['total'];
    }

    private function setFee()
    {
        $this->fee = 0;
    }

    private function setTotal()
    {
        $this->total = ceil(($this->due + $this->fee) * $this->convoRate);
    }

    private function logTransaction($payload, $message = 'Transaction Successful')
    {
        return logTransaction($this->gatewayModuleName, json_encode($payload), $message, "invoice", $this->invoice['id']);
    }

    private function addTransaction($trxId)
    {
        return addPayment($this->invoice['id'], $trxId, $this->due, $this->fee, $this->gatewayModuleName);
    }

    private function initPayment($requestData, $apiType = 'checkout-v2')
    {
        $apiUrl = $this->buildURL($apiType);
        $response = $this->sendRequest('POST', $apiUrl, $requestData);
        $this->validateApiResponse($response, 'Payment request failed');
        return $response;
    }

    private function verifyPayment($invoiceId)
    {
        $verifyUrl = $this->buildURL('verify-payment');
        $requestData = ['invoice_id' => $invoiceId];
        return $this->sendRequest('POST', $verifyUrl, $requestData);
    }

    private function executePayment()
    {
        $headerApi = $_SERVER['HTTP_RT_UDDOKTAPAY_API_KEY'] ?? null;
        $this->validateApiHeader($headerApi);

        $rawInput = trim(file_get_contents('php://input'));
        $this->validateIpnResponse($rawInput);

        $data = json_decode($rawInput, true);
        $invoiceId = $data['invoice_id'];

        return $this->verifyPayment($invoiceId);
    }

    private function sendRequest($method, $url, $data)
    {
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'RT-UDDOKTAPAY-API-KEY: ' . $this->credential['api_key'],
                'Accept: application/json',
                'Content-type: application/json',
            ],
        ]);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        curl_close($curl);

        if ($error) {
            throw new \Exception("Invalid response from UddoktaPay API.");
        }

        return json_decode($response, true);
    }

    private function validateApiHeader($headerApi)
    {
        if ($headerApi === null) {
            throw new \Exception("Invalid API Key");
        }

        $apiKey = trim($this->credential['api_key']);

        if ($headerApi !== $apiKey) {
            throw new \Exception("Unauthorized Action.");
        }
    }

    private function validateApiResponse($response, $errorMessage)
    {
        if (!isset($response['payment_url'])) {
            $message = isset($response['message']) ? $response['message'] : $errorMessage;
            throw new \Exception($message);
        }
    }

    private function validateIpnResponse($rawInput)
    {
        if (empty($rawInput)) {
            throw new \Exception("Invalid response from UddoktaPay API.");
        }
    }

    public function createPayment()
    {
        $callbackURL = $this->systemUrl . 'modules/gateways/callback/' . $this->gatewayModuleName . '.php?id=' . $this->invoice['id'] . '&action=ipn';
        $successURL = $this->systemUrl . 'modules/gateways/callback/' . $this->gatewayModuleName . '.php?id=' . $this->invoice['id'] . '&action=verify';
        $cancelURL = $this->systemUrl . _url("viewinvoice/id/" . md5($this->invoice['id'])) . "?errorCode=cancelled";
        $firstName = $this->client['first_name'];
        $lastName = $this->client['last_name'];
        $customerName = $firstName . ' ' . $lastName;
        $email = $this->client['email'];

        $fields = [
            "full_name" => $customerName,
            "email" => $email,
            "amount" => $this->total,
            "metadata" => [
                'invoice_id' => $this->invoice['id'],
            ],
            "redirect_url" => $successURL,
            "return_type" => "GET",
            "cancel_url" => $cancelURL,
            "webhook_url" => $callbackURL,
        ];

        try {
            $response = $this->initPayment($fields);

            if (is_array($response) && isset($response['payment_url'])) {
                return [
                    'status' => 'success',
                    'payment_url' => $response['payment_url'],
                ];
            }
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    public function makeTransaction($invoiceId, $ipn = false)
    {
        try {
            if ($ipn) {
                $executePayment = $this->executePayment();
            } else {
                $executePayment = $this->verifyPayment($invoiceId);
            }

            if (isset($executePayment['status']) && $executePayment['status'] === 'COMPLETED') {
                if (checkTransID($executePayment['transaction_id'])) {
                    return [
                        'status' => 'error',
                        'message' => 'The transaction has already been used.',
                    ];
                }

                if ($executePayment['amount'] < $this->total) {
                    return [
                        'status' => 'error',
                        'message' => 'You\'ve paid less than the required amount.',
                    ];
                }

                $this->logTransaction($executePayment);
                $this->addTransaction($executePayment['transaction_id']);

                return [
                    'status' => 'success',
                    'message' => 'The payment has been successfully verified.',
                ];
            } elseif (isset($executePayment['status']) && $executePayment['status'] === 'PENDING') {
                return [
                    'status' => 'error',
                    'message' => 'Your payment is pending for verification.',
                ];
            }
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }
}

$uddoktaPay = UddoktaPay::init();
if (!$uddoktaPay->isActive) {
    die("The gateway is unavailable.");
}

$action = $uddoktaPay->request['action'];
$invid = $uddoktaPay->request['id'];

if ($action === 'init') {
    $response = $uddoktaPay->createPayment();
    if ($response['status'] === 'success') {
        header('Location: ' . $response['payment_url']);
        exit;
    } else {
        header("Location: " . $uddoktaPay->systemUrl . _url("viewinvoice/id/" . md5($invid)) . "?errorCode={$response['message']}");
        exit;
    }
}
if ($action === 'verify') {
    $invoiceId = $uddoktaPay->request['invoice_id'];
    $response = $uddoktaPay->makeTransaction($invoiceId);

    if ($response['status'] === 'success') {
        header("Location: " . $uddoktaPay->systemUrl . _url("viewinvoice/id/" . md5($invid)));
        exit;
    } else {
        header("Location: " . $uddoktaPay->systemUrl . _url("viewinvoice/id/" . md5($invid)) . "?errorCode={$response['message']}");
        exit;
    }
}

if ($action === 'ipn') {
    $invoiceId = $uddoktaPay->request['invoice_id'];
    $response = $uddoktaPay->makeTransaction($invoiceId, true);
    exit;
}
header("Location: " . $uddoktaPay->systemUrl . _url("viewinvoice/id/" . md5($invid)) . "?errorCode=sww");
exit();
